<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Courses;
use App\Models\QuizOption;
use App\Models\QuizQuestion;
use Illuminate\Http\Request;

class QuizController extends Controller
{
    public function index(Request $request)
    {
        if ($request->isMethod('get')) {
            $title = 'Quiz Questions';
            $courses = Courses::where('is_deleted', 'N')
                ->where('is_active', '1')
                ->get();
            return view('admin.quiz.index', compact('title', 'courses'));
        }

        if ($request->isMethod('post')) {
            $data = QuizQuestion::with('course')->where('is_deleted', 'N')->where('is_active', '1')->orderBy('order_index', 'asc')->get();
            $records = [];
            $totalCount = count($data);

            if ($totalCount > 0) {
                foreach ($data as $key => $val) {

                    // Actions: Edit / Delete
                    $actions = '<div class="action-perform-btns">
                    <a class="btn p-0" href="' . route('quiz_question.edit', ['id' => $val->id]) . '" aria-label="Edit">
                        <i class="bi bi-pencil-square"></i>
                    </a>
                    <a table="quiz_questions" class="delete" href="javascript:void(0);" data-id="' . $val->id . '" aria-label="Delete">
                        <i class="bi bi-trash"></i>
                    </a>
                </div>';

                    $records[] = [
                        'SNO' => ++$key,
                        'COURSE' => $val->course?->title ?? '-',
                        'QUESTION' => $val->question_text,
                        'TYPE' => strtoupper(str_replace('_', ' ', $val->question_type)),
                        'POINTS' => $val->points,
                        'ORDER' => $val->order_index,
                        'ACTIVE' => $val->is_active ? 'Yes' : 'No',
                        'ACTIONS' => $actions,
                    ];
                }
            }

            $arraylist = [
                'TotalRecords' => $totalCount,
                'aaData' => $records,
            ];

            echo compressJson($arraylist);
        }
        
    }

    public function create(Request $request)
    {
        $title = 'Add Quiz';
        $courses = Courses::where('is_deleted', 'N')->where('has_quiz', '1')->get();

        return view('admin.quiz.add', compact('title', 'courses'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'course_id' => 'required|exists:courses,id',
            'question_text' => 'required|string',
            'question_type' => 'required|in:mcq_single,multi_select,true_false',
            'explanation_text' => 'nullable|string',
            'points' => 'nullable|integer|min:1',
            'order_index' => 'nullable|integer|min:1',

            'options' => 'required_if:question_type,mcq_single,multi_select|array|min:2',
            'options.*.option_text' => 'required|string',
            'options.*.feedback_if_selected' => 'nullable|string',

            'correct_single' => 'required_if:question_type,mcq_single|integer',

            'correct_tf' => 'required_if:question_type,true_false|integer',
        ]);
        $question = QuizQuestion::create([
            'course_id' => $request->course_id,
            'question_text' => $request->question_text,
            'question_type' => $request->question_type,
            'explanation_text' => $request->explanation_text,
            'points' => $request->points ?? 1,
            'order_index' => $request->order_index ?? 1,
            'is_active' => '1',
            'is_deleted' => 'N',
        ]);

        $optionsData = [];

        if ($request->question_type === 'true_false') {
            foreach ($request->options as $index => $opt) {
                $optionsData[] = [
                    'question_id' => $question->id,
                    'option_text' => $opt['option_text'],
                    'is_correct' => ((string) $request->is_correct === (string) $index),
                    'order_index' => $index + 1,
                    'is_active' => '1',
                    'is_deleted' => 'N',
                    'feedback_if_selected' => $opt['feedback_if_selected'] ?? null,
                ];
            }
        } else {
            foreach ($request->options as $index => $opt) {
                $isCorrect = false;
                if ($request->question_type === 'mcq_single') {
                    $isCorrect = ((string) $request->correct_single === (string) $index);
                } else {
                    $isCorrect = isset($opt['is_correct']);
                }

                $optionsData[] = [
                    'question_id' => $question->id,
                    'option_text' => $opt['option_text'],
                    'is_correct' => $isCorrect,
                    'order_index' => $index + 1,
                    'is_active' => '1',
                    'is_deleted' => 'N',
                    'feedback_if_selected' => $opt['feedback_if_selected'] ?? null,
                ];
            }
        }

        if (!empty($optionsData)) {
            QuizOption::insert($optionsData);
        }

        if ($question) {
            return redirect()->route('quiz_question.index')->with('success', 'Quiz Question added successfully');
        }

        return back()->with('error', 'Something went wrong')->withInput();
    }

    public function edit($id)
    {
        $title = 'Edit Quiz';
        $courses = Courses::where('is_deleted', 'N')->where('has_quiz', '1')->get();
        $question = QuizQuestion::with('options')->where('is_deleted', 'N')->findOrFail($id);

        return view('admin.quiz.edit', compact('title', 'courses', 'question'));
    }

    public function update(Request $request, $id)
    {
        $question = QuizQuestion::where('is_deleted', 'N')->findOrFail($id);

        $request->validate([
            'course_id' => 'required|exists:courses,id',
            'question_text' => 'required|string',
            'question_type' => 'required|in:mcq_single,multi_select,true_false',
            'explanation_text' => 'nullable|string',
            'points' => 'nullable|integer|min:1',
            'order_index' => 'nullable|integer|min:1',

            'options' => 'required_if:question_type,mcq_single,multi_select|array|min:2',
            'options.*.option_text' => 'required|string',
            'options.*.feedback_if_selected' => 'nullable|string',

            'correct_single' => 'required_if:question_type,mcq_single|integer',
            'correct_tf' => 'required_if:question_type,true_false|integer',
        ]);

        $question->update([
            'course_id' => $request->course_id,
            'question_text' => $request->question_text,
            'question_type' => $request->question_type,
            'explanation_text' => $request->explanation_text,
            'points' => $request->points ?? 1,
            'order_index' => $request->order_index ?? 1,
        ]);

        QuizOption::where('question_id', $question->id)
            ->update([
                'is_deleted' => 'Y',
                'updated_at' => now()
            ]);


        $optionsData = [];
        if ($request->question_type === 'true_false') {
            foreach ($request->options as $i => $opt) {
                $optionsData[] = [
                    'question_id' => $question->id,
                    'option_text' => $opt['option_text'],
                    'is_correct' => ((string) $request->is_correct === (string) $i),
                    'order_index' => $i + 1,
                    'is_active' => 1,
                    'is_deleted' => 'N',
                    'feedback_if_selected' => $opt['feedback_if_selected'] ?? null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }
        } else {
            foreach ($request->options as $i => $opt) {
                $isCorrect = false;
                if ($request->question_type === 'mcq_single') {
                    $isCorrect = ((string) $request->correct_single === (string) $i);
                } else {
                    $isCorrect = isset($opt['is_correct']);
                }
                $optionsData[] = [
                    'question_id' => $question->id,
                    'option_text' => $opt['option_text'],
                    'is_correct' => $isCorrect,
                    'order_index' => $i + 1,
                    'is_active' => 1,
                    'is_deleted' => 'N',
                    'feedback_if_selected' => $opt['feedback_if_selected'] ?? null,
                ];
            }
        }

        if (!empty($optionsData)) {
            QuizOption::insert($optionsData);
        }

        return redirect()->route('quiz_question.index')->with('success', 'Quiz Question updated successfully');
    }
    
}
