<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CourseFiles;
use App\Models\Courses;
use App\Models\Programs;
use Exception;
use Illuminate\Http\Request;

class CourseController extends Controller
{
    public function index(Request $request)
    {
        if ($request->isMethod('get')) {
            $title = 'Courses';

            return view('admin.courses.index', compact('title'));
        }

        if ($request->isMethod('post')) {
            $data = Courses::where('is_deleted', 'N')->with('program')->orderBy('order_index', 'asc')->get();

            $records = [];
            $totalCount = count($data);

            if ($totalCount > 0) {
                foreach ($data as $key => $val) {

                    $actions = '<div class="action-perform-btns">
                        <a class="btn p-0" href="'.route('courses.edit', ['id' => $val->id]).'" aria-label="Edit" data-bs-original-title="Edit">
                            <i class="bi bi-pencil-square"></i>
                        </a>
                        <a table="courses" class="delete" href="javascript:void(0);" data-id="'.$val->id.'" aria-label="Delete" data-bs-original-title="Delete">
                            <i class="bi bi-trash"></i>
                        </a>
                    </div>';

                    $records[] = [
                        'SNO' => ++$key,
                        'PROGRAM' => $val->program?->title ?? '-',
                        'TITLE' => $val->title,
                        'ORDER' => $val->order_index,
                        'HAS_QUIZ' => $val->has_quiz ? 'Yes' : 'No',
                        'UNLOCK' => $val->unlock_type,
                        'ACTIONS' => $actions,
                    ];
                }
            }

            $arraylist = [
                'TotalRecords' => $totalCount,
                'aaData' => $records,
            ];

            echo compressJson($arraylist);
        }
    }

    // GET Add Page
    public function create()
    {
        $title = 'Add Course';
        $programs = Programs::where('is_deleted', 'N')->where('is_active', '1')->get();
        $courses = Courses::where('is_deleted', 'N')->get(); // for prerequisite dropdown

        return view('admin.courses.add', compact('title', 'programs', 'courses'));
    }

    // POST Create Logic
    public function store(Request $request)
    {
        $request->validate([
            'program_id' => 'nullable|exists:programs,id',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'order_index' => 'required|integer|min:1',

            'video_url' => 'nullable|string',
            'slides_url' => 'nullable|string',
            'transcript' => 'nullable|string',

            'has_quiz' => 'nullable|boolean',
            'passing_percent' => 'nullable|integer|min:0|max:100',
            'allow_unlimited_attempts' => 'nullable|boolean',
            'max_attempts' => 'nullable|integer|min:1',

            'unlock_type' => 'required|in:open,after_prerequisite',
            'prerequisite_course_id' => 'nullable|exists:courses,id',
            'prerequisite_condition' => 'required|in:completed,passed',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|max:10240|mimes:pdf,ppt,pptx,doc,docx,xls,xlsx',
        ]);

        $hasQuiz = $request->boolean('has_quiz');
        $allowUnlimited = $request->boolean('allow_unlimited_attempts', true);

        $course = Courses::create([
            'program_id' => $request->program_id,
            'title' => $request->title,
            'description' => $request->description,
            'order_index' => $request->order_index,

            'video_url' => $request->video_url,
            'slides_url' => $request->slides_url,
            'transcript' => $request->transcript,

            'has_quiz' => $hasQuiz ? 1 : 0,
            'passing_percent' => $hasQuiz ? ($request->passing_percent ?? 95) : null,
            'allow_unlimited_attempts' => $allowUnlimited ? 1 : 0,
            'max_attempts' => $allowUnlimited ? null : $request->max_attempts,

            'unlock_type' => $request->unlock_type,
            'prerequisite_course_id' => $request->unlock_type === 'after_prerequisite' ? $request->prerequisite_course_id : null,
            'prerequisite_condition' => $request->prerequisite_condition,

            'is_active' => '1',
            'is_deleted' => 'N',
        ]);

        if ($course) {
            if ($request->hasFile('attachments')) {
                foreach ($request->file('attachments') as $file) {
                    $path = $file->store('courses/files', 'public');

                    CourseFiles::create([
                        'course_id' => $course->id,
                        'file_name' => $file->getClientOriginalName(),
                        'file_path' => $path,
                        'file_type' => $file->getClientOriginalExtension(),
                    ]);
                }
            }

            return redirect()->route('courses.index')->with('success', 'Course added successfully');
        }

        return back()->with('error', 'Something went wrong')->withInput();
    }

    // GET Edit Page
    public function edit($id)
    {
        $title = 'Edit Course';
        $course = Courses::where('is_deleted', 'N')->findOrFail($id);
        $programs = Programs::where('is_deleted', 'N')->where('is_active', '1')->get();
        $courses = Courses::where('is_deleted', 'N')->where('id', '!=', $id)->get(); // exclude itself

        return view('admin.courses.edit', compact('title', 'course', 'programs', 'courses'));
    }

    // POST Update Logic
    public function update(Request $request, $id)
    {
        try {
            $course = Courses::where('is_deleted', 'N')->findOrFail($id);

            $request->validate([
                'program_id' => 'nullable|exists:programs,id',
                'title' => 'required|string|max:255',
                'description' => 'nullable|string',
                'order_index' => 'required|integer|min:1',

                'video_url' => 'nullable|string',
                'slides_url' => 'nullable|string',
                'transcript' => 'nullable|string',

                'has_quiz' => 'nullable|boolean',
                'passing_percent' => 'nullable|integer|min:0|max:100',
                'allow_unlimited_attempts' => 'nullable|boolean',
                'max_attempts' => 'nullable|integer|min:1',

                'unlock_type' => 'required|in:open,after_prerequisite',
                'prerequisite_course_id' => 'nullable|exists:courses,id',
                'prerequisite_condition' => 'required|in:completed,passed',
                'attachments' => 'nullable|array',
                'attachments.*' => 'file|max:10240|mimes:pdf,ppt,pptx,doc,docx,xls,xlsx',
            ]);

            $hasQuiz = $request->boolean('has_quiz');
            $allowUnlimited = $request->boolean('allow_unlimited_attempts', true);

            $course->program_id = $request->program_id;
            $course->title = $request->title;
            $course->description = $request->description;
            $course->order_index = $request->order_index;

            $course->video_url = $request->video_url;
            $course->slides_url = $request->slides_url;
            $course->transcript = $request->transcript;

            $course->has_quiz = $hasQuiz ? 1 : 0;
            $course->passing_percent = $hasQuiz ? ($request->passing_percent ?? 95) : null;
            $course->allow_unlimited_attempts = $allowUnlimited ? 1 : 0;
            $course->max_attempts = $allowUnlimited ? null : $request->max_attempts;

            $course->unlock_type = $request->unlock_type;
            $course->prerequisite_course_id = $request->unlock_type === 'after_prerequisite' ? $request->prerequisite_course_id : null;
            $course->prerequisite_condition = $request->prerequisite_condition;

            $query = $course->save();

            if ($query) {
                if ($request->hasFile('attachments')) {
                    foreach ($request->file('attachments') as $file) {
                        $path = $file->store('courses/files', 'public');

                        CourseFiles::create([
                            'course_id' => $course->id,
                            'file_name' => $file->getClientOriginalName(),
                            'file_path' => $path,
                            'file_type' => $file->getClientOriginalExtension(),
                        ]);
                    }
                }

                return redirect()->route('courses.index')->with('success', 'Course updated successfully');
            }

            return back()->with('error', 'Something went wrong');
        } catch (Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }
       

}
